const { app, BrowserWindow, ipcMain } = require("electron");
const { spawn } = require("child_process");
const fs = require("fs");
const path = require("path");

const logFilePath = path.join(__dirname, "logs.txt");

// Clear log file on startup
fs.writeFileSync(logFilePath, "", "utf8");

let terminal;

/**
 * Open a system terminal and follow logs.txt
 */
function spawnTerminal() {
  let cmd;
  let args;

  if (process.platform === "win32") {
    // Windows: PowerShell
    cmd = "cmd.exe";
    args = [
      "/c",
      "start",
      "powershell",
      "-NoExit",
      `Get-Content -Path "${logFilePath}" -Wait`
    ];
  } else if (process.platform === "darwin") {
    // macOS: Terminal.app
    cmd = "osascript";
    args = [
      "-e",
      `tell application "Terminal"
        activate
        do script "clear && tail -f '${logFilePath}'"
      end tell`
    ];
  } else if (process.platform === "linux") {
    // Linux: try common terminals
    const terminals = [
      ["x-terminal-emulator", ["-e", `tail -f "${logFilePath}"`]],
      ["gnome-terminal", ["--", "tail", "-f", logFilePath]],
      ["konsole", ["-e", "tail", "-f", logFilePath]],
      ["xfce4-terminal", ["-e", `tail -f "${logFilePath}"`]]
    ];

    for (const [tCmd, tArgs] of terminals) {
      try {
        terminal = spawn(tCmd, tArgs, { stdio: "ignore", detached: true });
        terminal.unref();
        return;
      } catch (_) {}
    }

    console.warn("No supported terminal found on Linux.");
    return;
  } else {
    console.warn("Unsupported platform:", process.platform);
    return;
  }

  terminal = spawn(cmd, args, { stdio: "ignore", detached: true });

  terminal.on("error", (err) => {
    console.error("Failed to open terminal:", err);
  });
}

/**
 * Mirror console.log into logs.txt
 */
function overrideConsoleLog() {
  const originalLog = console.log;

  console.log = (...args) => {
    const line = args.map(String).join(" ") + "\n";
    fs.appendFileSync(logFilePath, line, "utf8");
    originalLog(...args);
  };
}

/**
 * Create Electron window
 */
function createWindow() {
  const win = new BrowserWindow({
    width: 800,
    height: 600,
    fullscreen: true,
    show: false,
    autoHideMenuBar: true,
    title: "Vizer client",
    webPreferences: {
      preload: path.join(__dirname, "preload.js"),
      webSecurity: false,
      backgroundThrottling: false
    }
  });

win.loadFile(path.join(__dirname, "client.html"));


  win.once("ready-to-show", () => win.show());

  win.on("closed", () => {
    if (terminal && !terminal.killed) {
      terminal.kill();
    }
  });
}

/**
 * App lifecycle
 */
app.whenReady().then(() => {
  overrideConsoleLog();
  createWindow();
  spawnTerminal();
});

app.on("window-all-closed", () => {
  if (terminal && !terminal.killed) {
    terminal.kill();
  }

  if (process.platform !== "darwin") {
    app.quit();
  }
});

/**
 * IPC logging from renderer
 */
ipcMain.on("toMain", (_event, message) => {
  const now = new Date();
  const ts =
    `[${now.getFullYear()}-${String(now.getMonth() + 1).padStart(2, "0")}-${String(now.getDate()).padStart(2, "0")} ` +
    `${String(now.getHours()).padStart(2, "0")}:${String(now.getMinutes()).padStart(2, "0")}:${String(now.getSeconds()).padStart(2, "0")}]`;

  console.log(`${ts}\t${message}`);
});
